﻿import { Router, Request, Response } from 'express';
import { prisma } from '../../lib/prisma.js';
import { z } from 'zod';

const router = Router();

// Validation schemas
const movieQuerySchema = z.object({
  page: z.string().optional().default('1'),
  pageSize: z.string().optional().default('10'),
  status: z.string().optional(),
  genre: z.string().optional(),
  search: z.string().optional()
});

// Get all movies with pagination and filtering
router.get('/', async (req: Request, res: Response) => {
  try {
    const { page, pageSize, status, genre, search } = movieQuerySchema.parse(req.query);
    
    const pageNum = parseInt(page);
    const pageSizeNum = parseInt(pageSize);
    const skip = (pageNum - 1) * pageSizeNum;

    // Build where clause
    const where: any = {};
    
    if (search) {
      where.OR = [
        { title: { contains: search, mode: 'insensitive' } },
        { synopsis: { contains: search, mode: 'insensitive' } }
      ];
    }
    
    if (genre) {
      where.genre = { contains: genre, mode: 'insensitive' };
    }

    // Get movies with showtimes for status filtering
    const movies = await prisma.movie.findMany({
      where,
      include: {
        showtimes: {
          where: status ? {
            showDate: status === 'now-showing' 
              ? { gte: new Date() }
              : status === 'coming-soon'
              ? { gte: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000) }
              : {}
          } : {},
          include: {
            hall: true
          }
        },
        reviews: {
          select: {
            rating: true
          }
        },
        _count: {
          select: {
            reviews: true
          }
        }
      },
      skip,
      take: pageSizeNum,
      orderBy: {
        ratingAvg: 'desc'
      }
    });

    // Filter movies based on status if showtimes exist
    let filteredMovies = movies;
    if (status === 'now-showing') {
      filteredMovies = movies.filter(movie => 
        movie.showtimes.some(showtime => showtime.showDate >= new Date())
      );
    } else if (status === 'coming-soon') {
      filteredMovies = movies.filter(movie => 
        movie.showtimes.some(showtime => showtime.showDate >= new Date(Date.now() + 7 * 24 * 60 * 60 * 1000))
      );
    }

    // Calculate average ratings and convert basePrice to number
    const moviesWithRatings = filteredMovies.map(movie => ({
      ...movie,
      ratingAvg: Number(movie.ratingAvg),
      averageRating: movie.reviews.length > 0 
        ? movie.reviews.reduce((sum, review) => sum + review.rating, 0) / movie.reviews.length
        : Number(movie.ratingAvg),
      reviewCount: movie._count.reviews,
      showtimes: movie.showtimes.map(showtime => ({
        ...showtime,
        basePrice: Number(showtime.basePrice)
      }))
    }));

    // Get total count for pagination
    const total = await prisma.movie.count({ where });

    res.json({
      success: true,
      data: {
        items: moviesWithRatings,
        page: pageNum,
        pageSize: pageSizeNum,
        total,
        totalPages: Math.ceil(total / pageSizeNum)
      }
    });

  } catch (error) {
    console.error('Error fetching movies:', error);
    
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: 'Validation error',
        errors: error.issues
      });
    }

    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// Get movie by ID
router.get('/:id', async (req: Request, res: Response) => {
  try {
    const movieId = parseInt(req.params.id);
    
    if (isNaN(movieId)) {
      return res.status(400).json({
        success: false,
        message: 'Invalid movie ID'
      });
    }

    const movie = await prisma.movie.findUnique({
      where: { id: movieId },
      include: {
        showtimes: {
          include: {
            hall: true
          },
          orderBy: {
            showDate: 'asc'
          }
        },
        reviews: {
          include: {
            user: {
              select: {
                name: true
              }
            }
          },
          orderBy: {
            createdAt: 'desc'
          }
        },
        _count: {
          select: {
            reviews: true
          }
        }
      }
    });

    if (!movie) {
      return res.status(404).json({
        success: false,
        message: 'Movie not found'
      });
    }

    // Calculate average rating and convert numeric strings to numbers
    const averageRating = movie.reviews.length > 0 
      ? movie.reviews.reduce((sum, review) => sum + review.rating, 0) / movie.reviews.length
      : Number(movie.ratingAvg);

    const movieWithRating = {
      ...movie,
      ratingAvg: Number(movie.ratingAvg),
      averageRating,
      reviewCount: movie._count.reviews,
      showtimes: movie.showtimes.map(showtime => ({
        ...showtime,
        basePrice: Number(showtime.basePrice)
      }))
    };

    res.json({
      success: true,
      data: { movie: movieWithRating }
    });

  } catch (error) {
    console.error('Error fetching movie:', error);
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// Get featured/trending movies
router.get('/featured/list', async (req: Request, res: Response) => {
  try {
    const movies = await prisma.movie.findMany({
      include: {
        showtimes: {
          where: {
            showDate: { gte: new Date() }
          },
          include: {
            hall: true
          }
        },
        reviews: {
          select: {
            rating: true
          }
        },
        _count: {
          select: {
            reviews: true
          }
        }
      },
      orderBy: {
        ratingAvg: 'desc'
      },
      take: 6 // Top 6 featured movies
    });

    const moviesWithRatings = movies.map(movie => ({
      ...movie,
      ratingAvg: Number(movie.ratingAvg),
      averageRating: movie.reviews.length > 0 
        ? movie.reviews.reduce((sum, review) => sum + review.rating, 0) / movie.reviews.length
        : Number(movie.ratingAvg),
      reviewCount: movie._count.reviews,
      showtimes: movie.showtimes.map(showtime => ({
        ...showtime,
        basePrice: Number(showtime.basePrice)
      }))
    }));

    res.json({
      success: true,
      data: { movies: moviesWithRatings }
    });

  } catch (error) {
    console.error('Error fetching featured movies:', error);
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// Get movie genres
router.get('/genres/list', async (req: Request, res: Response) => {
  try {
    const movies = await prisma.movie.findMany({
      select: {
        genre: true
      }
    });

    // Extract unique genres
    const genres = new Set<string>();
    movies.forEach(movie => {
      if (movie.genre) {
        movie.genre.split(',').forEach(genre => {
          genres.add(genre.trim());
        });
      }
    });

    res.json({
      success: true,
      data: { genres: Array.from(genres) }
    });

  } catch (error) {
    console.error('Error fetching genres:', error);
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

export default router;