import { PrismaClient } from '@prisma/client';
import bcrypt from 'bcryptjs';
const prisma = new PrismaClient();
async function seedData() {
    try {
        console.log('🌱 Starting database seeding...');
        // Clear existing data
        await prisma.booking.deleteMany();
        await prisma.review.deleteMany();
        await prisma.showtime.deleteMany();
        await prisma.seat.deleteMany();
        await prisma.hall.deleteMany();
        await prisma.movie.deleteMany();
        await prisma.user.deleteMany();
        console.log('🗑️ Cleared existing data');
        // Create sample users
        const hashedPassword = await bcrypt.hash('password123', 10);
        const users = await prisma.user.createMany({
            data: [
                {
                    name: 'John Doe',
                    email: 'john@example.com',
                    phone: '+1234567890',
                    password: hashedPassword
                },
                {
                    name: 'Jane Smith',
                    email: 'jane@example.com',
                    phone: '+1234567891',
                    password: hashedPassword
                },
                {
                    name: 'Admin User',
                    email: 'admin@cinema.com',
                    phone: '+1234567892',
                    password: hashedPassword
                }
            ]
        });
        console.log('👥 Created sample users');
        // Create sample movies
        const movies = await prisma.movie.createMany({
            data: [
                {
                    title: 'The Dark Knight',
                    genre: 'Action, Crime, Drama',
                    durationMin: 152,
                    posterUrl: 'https://images.unsplash.com/photo-1489599808427-2a2b5b5b5b5b?w=300',
                    trailerUrl: 'https://www.youtube.com/watch?v=EXeTwQWrcwY',
                    synopsis: 'When the menace known as the Joker wreaks havoc and chaos on the people of Gotham, Batman must accept one of the greatest psychological and physical tests of his ability to fight injustice.',
                    ratingAvg: 9.0
                },
                {
                    title: 'Inception',
                    genre: 'Action, Sci-Fi, Thriller',
                    durationMin: 148,
                    posterUrl: 'https://images.unsplash.com/photo-1489599808427-2a2b5b5b5b5b?w=300',
                    trailerUrl: 'https://www.youtube.com/watch?v=YoHD9XEInc0',
                    synopsis: 'A thief who steals corporate secrets through the use of dream-sharing technology is given the inverse task of planting an idea into the mind of a C.E.O.',
                    ratingAvg: 8.8
                },
                {
                    title: 'Interstellar',
                    genre: 'Adventure, Drama, Sci-Fi',
                    durationMin: 169,
                    posterUrl: 'https://images.unsplash.com/photo-1489599808427-2a2b5b5b5b5b?w=300',
                    trailerUrl: 'https://www.youtube.com/watch?v=zSWdZVtXT7E',
                    synopsis: 'A team of explorers travel through a wormhole in space in an attempt to ensure humanity\'s survival.',
                    ratingAvg: 8.6
                },
                {
                    title: 'The Matrix',
                    genre: 'Action, Sci-Fi',
                    durationMin: 136,
                    posterUrl: 'https://images.unsplash.com/photo-1489599808427-2a2b5b5b5b5b?w=300',
                    trailerUrl: 'https://www.youtube.com/watch?v=vKQi3bBA1y8',
                    synopsis: 'A computer hacker learns from mysterious rebels about the true nature of his reality and his role in the war against its controllers.',
                    ratingAvg: 8.7
                },
                {
                    title: 'Pulp Fiction',
                    genre: 'Crime, Drama',
                    durationMin: 154,
                    posterUrl: 'https://images.unsplash.com/photo-1489599808427-2a2b5b5b5b5b?w=300',
                    trailerUrl: 'https://www.youtube.com/watch?v=s7EdQ4FqbhY',
                    synopsis: 'The lives of two mob hitmen, a boxer, a gangster and his wife, and a pair of diner bandits intertwine in four tales of violence and redemption.',
                    ratingAvg: 8.9
                },
                {
                    title: 'The Shawshank Redemption',
                    genre: 'Drama',
                    durationMin: 142,
                    posterUrl: 'https://images.unsplash.com/photo-1489599808427-2a2b5b5b5b5b?w=300',
                    trailerUrl: 'https://www.youtube.com/watch?v=6hB3S9bIaco',
                    synopsis: 'Two imprisoned men bond over a number of years, finding solace and eventual redemption through acts of common decency.',
                    ratingAvg: 9.3
                }
            ]
        });
        console.log('🎬 Created sample movies');
        // Get created movies for showtimes
        const createdMovies = await prisma.movie.findMany();
        // Create cinema halls
        const halls = await prisma.hall.createMany({
            data: [
                {
                    name: 'Hall A - Premium',
                    capacity: 100
                },
                {
                    name: 'Hall B - Standard',
                    capacity: 150
                },
                {
                    name: 'Hall C - IMAX',
                    capacity: 200
                },
                {
                    name: 'Hall D - VIP',
                    capacity: 50
                }
            ]
        });
        console.log('🏛️ Created cinema halls');
        // Get created halls
        const createdHalls = await prisma.hall.findMany();
        // Create seats for each hall
        for (const hall of createdHalls) {
            const seats = [];
            const rows = Math.ceil(hall.capacity / 20); // 20 seats per row
            for (let row = 1; row <= rows; row++) {
                const seatsInRow = row === rows ? hall.capacity - (row - 1) * 20 : 20;
                for (let seat = 1; seat <= seatsInRow; seat++) {
                    seats.push({
                        hallId: hall.id,
                        label: `${String.fromCharCode(64 + row)}${seat.toString().padStart(2, '0')}`
                    });
                }
            }
            await prisma.seat.createMany({
                data: seats
            });
        }
        console.log('🪑 Created seats for all halls');
        // Create showtimes for the next 7 days
        const showtimes = [];
        const today = new Date();
        for (let day = 0; day < 7; day++) {
            const showDate = new Date(today);
            showDate.setDate(today.getDate() + day);
            // Create 3-4 showtimes per day for each movie
            for (const movie of createdMovies) {
                const timeSlots = ['10:00', '14:00', '18:00', '22:00'];
                for (let i = 0; i < Math.min(3, timeSlots.length); i++) {
                    const [hours, minutes] = timeSlots[i].split(':');
                    const showDateTime = new Date(showDate);
                    showDateTime.setHours(parseInt(hours), parseInt(minutes), 0, 0);
                    // Random hall assignment
                    const randomHall = createdHalls[Math.floor(Math.random() * createdHalls.length)];
                    // Different pricing based on hall type
                    let basePrice = 12.00;
                    if (randomHall.name.includes('Premium'))
                        basePrice = 15.00;
                    if (randomHall.name.includes('IMAX'))
                        basePrice = 18.00;
                    if (randomHall.name.includes('VIP'))
                        basePrice = 25.00;
                    showtimes.push({
                        movieId: movie.id,
                        hallId: randomHall.id,
                        showDate: showDateTime,
                        basePrice: basePrice
                    });
                }
            }
        }
        await prisma.showtime.createMany({
            data: showtimes
        });
        console.log('🎭 Created showtimes');
        // Create some sample reviews
        const reviews = [];
        for (const movie of createdMovies) {
            const numReviews = Math.floor(Math.random() * 5) + 3; // 3-7 reviews per movie
            for (let i = 0; i < numReviews; i++) {
                const randomUser = await prisma.user.findFirst({
                    skip: Math.floor(Math.random() * 3)
                });
                if (randomUser) {
                    reviews.push({
                        movieId: movie.id,
                        userId: randomUser.id,
                        rating: Math.floor(Math.random() * 5) + 1, // 1-5 stars
                        comment: `Great movie! ${movie.title} was amazing. Highly recommended!`
                    });
                }
            }
        }
        await prisma.review.createMany({
            data: reviews
        });
        console.log('⭐ Created sample reviews');
        console.log('✅ Database seeding completed successfully!');
        console.log(`📊 Created:`);
        console.log(`   - ${await prisma.user.count()} users`);
        console.log(`   - ${await prisma.movie.count()} movies`);
        console.log(`   - ${await prisma.hall.count()} halls`);
        console.log(`   - ${await prisma.seat.count()} seats`);
        console.log(`   - ${await prisma.showtime.count()} showtimes`);
        console.log(`   - ${await prisma.review.count()} reviews`);
    }
    catch (error) {
        console.error('❌ Error seeding database:', error);
        throw error;
    }
    finally {
        await prisma.$disconnect();
    }
}
// Run the seeding
seedData()
    .then(() => {
    console.log('🎉 Seeding completed!');
    process.exit(0);
})
    .catch((error) => {
    console.error('💥 Seeding failed:', error);
    process.exit(1);
});
